
/******************************************************************************

Copyright (c) 2007, Niversoft Ides Logicielles
All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, 
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright 
      notice, this list of conditions and the following disclaimer in the 
      documentation and/or other materials provided with the distribution.
    * Neither the name of Niversoft Ides Logicielles nor the names of its 
      contributors may be used to endorse or promote products derived from 
      this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE 
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

******************************************************************************/

#pragma once

using namespace System;
using namespace System::Text;
using namespace System::Net::Sockets;
using namespace System::Security::Cryptography;

typedef unsigned short word;

namespace CommunigatePro 
{

   public ref class CLIException : System::Exception
   {
   public:
      CLIException(String ^message, int code) : Exception(code.ToString() + ": " + message) {}
      CLIException(String ^message) : Exception(message) {}      
   };

   public ref class InvalidLoginException : CLIException
   {
   public:
      InvalidLoginException(String ^message) : CLIException(message) {}
   };

   public ref class ProtocolException : CLIException
   {
   public:
      ProtocolException(String ^message) : CLIException(message) {}
   };

   public ref class InvalidServerResponseException : CLIException
   {
   public:
      InvalidServerResponseException(String ^message) : CLIException(message) {}
   };

   public ref class InvalidServerPromptException : CLIException
   {
   public:
      InvalidServerPromptException(String ^message) : CLIException(message) {}
   };   
   
	public ref class CLI
	{
   public:
      static bool     DefaultSecureLogin = true;
      static String ^ DefaultIp = "127.0.0.1";
      static word     DefaultPort = 106;
      static String ^ DefaultUser = "";
      static String ^ DefaultPassword = "";

   private:
      TcpClient ^          m_Client;
      NetworkStream ^      m_Stream;
      String ^             m_Data;
      String ^             m_Command;
      int                  m_Len;
      int                  m_Span;
      word                 m_ErrCode;
      String ^             m_ErrMsg;
      String ^             m_ResponseLine;
      String ^             InlineResponse;

      String ^ Read()
      {         
         String^ myCompleteMessage = "";
         if ( m_Stream->CanRead )
         {
            array<Byte>^ myReadBuffer = gcnew array<Byte>(1024);
            int numberOfBytesRead = 0;
            
            do
            {
               numberOfBytesRead = m_Stream->Read( myReadBuffer, 0,
                  myReadBuffer->Length );
               myCompleteMessage = String::Concat( myCompleteMessage,
                  Encoding::ASCII->GetString( myReadBuffer, 0, numberOfBytesRead ) );
            }
            while ( !myCompleteMessage->EndsWith("\n") );
         }
         return myCompleteMessage;
      }

      void Write(String ^ in_msg)
      {
         array<Byte> ^ l_Buf = Encoding::ASCII->GetBytes(in_msg + "\n");         
         m_Stream->Write(l_Buf, 0, l_Buf->Length);
      }

      void StreamSend(String ^ in_Command)
      {
         Write(in_Command);
      }

      void SendCmd(String ^ in_Command)
      {
         Write(in_Command);
         ParseResponse();
      }


      void ParseResponse()
      {
         ParseResponse(true);
      }

      void ParseResponse(bool in_ThrowOnErr)
      {
         m_ResponseLine = Read();
         int spacePos = m_ResponseLine->IndexOf(' ');
         if(spacePos > 0)
         {
            m_ErrCode = (word)Convert::ToInt32(m_ResponseLine->Substring(0, spacePos));
            if (m_ErrCode == 0)
            {
               throw gcnew ProtocolException(m_ResponseLine);         
            }
            if ((m_ErrCode == 200) || (m_ErrCode == 201))
            {
               InlineResponse = m_ResponseLine->Substring(spacePos + 1);
               m_ErrMsg = "OK";
            }
            else
            {
               m_ErrMsg = m_ResponseLine->Substring(spacePos + 1);               
               if (m_ErrMsg->EndsWith("\r\n"))
               {                  
                  m_ErrMsg = m_ErrMsg->Substring(0, m_ErrMsg->Length - 2);
               }
               if (in_ThrowOnErr)
               {
                  throw gcnew CLIException(m_ErrMsg->Trim(), m_ErrCode);
               }
            }
         }
         else
         {            
            throw gcnew InvalidServerResponseException(m_ResponseLine->Trim());      
         }
      }

      String ^ GetWords()
      {
         if(m_ErrCode == 201)
            return InlineResponse;

         return Read();
      }

      void SkipSpaces()
      {
         for(; m_Span < m_Len && m_Data[m_Span] <= ' '; m_Span++);
      }

      String ^ ReadWord()
      {
         bool isQuoted = false;
         bool isBlock = false;
         StringBuilder ^ sbResult = gcnew StringBuilder();
         
         SkipSpaces();
         if (m_Data[m_Span] == '"')
         {
            isQuoted = true;
            m_Span++;
         }
         else if(m_Data[m_Span] == '[')
         {
            isBlock = true;
         }

         while(m_Span < m_Len)
         {
            wchar_t ch = m_Data[m_Span];
            if(isQuoted)
            {
               if(ch == '"')
               {
                  m_Span++;
                  break;
               }
               if(ch == '\\' && m_Span + 1 < m_Len)
               {
                  wchar_t ch2 = m_Data[m_Span + 1];
                  if(ch2 == '"' || ch2 == '\\')
                  {                     
                     sbResult->Append(ch,1);
                     sbResult->Append(ch2,1);
                     m_Span += 2;
                     continue;
                  }
               }
            }
            else
               if(isBlock)
               {
                  if(ch == ']')
                  {
                     sbResult->Append(ch,1);
                     m_Span++;
                     break;
                  }
               }
               else if((ch < 'a' || ch > 'z') && (ch < 'A' || ch > 'Z') && (ch < '0' || ch > '9') && ch != '_' && ch != '.' && ch != '@' && ch != '!' && ch != '#' && ch != '%' && ch != '-')
                  break;
            sbResult->Append(ch,1);
            m_Span++;
         }
         return sbResult->ToString();
      }

      DataSet ^ ReadValue()
      {
         SkipSpaces();
         char ch = (char)m_Data[m_Span];
         if(ch == '{')
         {
            m_Span++;
            return ReadDictionary();
         }
         if(ch == '(')
         {
            m_Span++;
            return ReadArray();
         }
         else
         {
            return DataSet::String(ReadWord());
         }
      }

      DataSet ^ ReadArray()
      {
         DataSet ^ result = gcnew DataSet(DataSet::DataSetType::Array);
         while(m_Span < m_Len)
         {
            SkipSpaces();
            if(m_Data[m_Span] == ')')
            {
               m_Span++;
               break;
            }
            DataSet ^ theValue = ReadValue();
            SkipSpaces();
            result->ArrayValue->Add(theValue);
            if(m_Data[m_Span] == ',')
               m_Span++;
            else if(m_Data[m_Span] != ')')
            {
               m_ErrMsg = "CGPro output format Error 1:" + m_Data->Substring(m_Span);
               m_ErrCode = -1;                              
            }
         }
         return result;
      }

      DataSet ^ ReadDictionary()
      {
         DataSet ^ result = gcnew DataSet(DataSet::DataSetType::Dictionary);
         for(; m_Span < m_Len; m_Span++)
         {
            SkipSpaces();
            if(m_Data[m_Span] == '}')
            {
               m_Span++;
               break;
            }
            String ^ theKey = ReadWord();
            SkipSpaces();
            if(m_Data[m_Span] != '=')
            {
               m_ErrMsg = "CGPro output format Error 2:" + m_Data->Substring(m_Span);
               m_ErrCode = -2;
            }
            m_Span++;
            result->DictionaryValue->Add(theKey, ReadValue());
            SkipSpaces();
            if(m_Data[m_Span] != ';')
            {
               m_ErrMsg = "CGPro output format Error 3:" + m_Data->Substring(m_Span);
               m_ErrCode = -3;                              
            }
         }

         return result;
      }

      DataSet ^ ParseWords(String ^ s)
      {
         m_Len = s->Length;
         m_Span = 0;
         m_Data = s;
         return ReadValue();
      }

      bool MustQuote(String ^ s)
      {
         if (s->Length == 0)
            return true;

         String ^ ValidChars = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789_@.-";
         CharEnumerator ^c = s->GetEnumerator(); 
         while (c->MoveNext())
         {
            if (ValidChars->IndexOf(c->Current) == -1)
               return true;
         }
         return false;            
      }

      String ^ Encode(String ^ obj)
      {
         if (MustQuote(obj))
            return "\"" + obj + "\"";
         else
            return obj;
      }

      String ^ Encode(DataSet ^ obj)
      {
         StringBuilder ^ s = gcnew StringBuilder();
         if(obj->Type() == DataSet::DataSetType::Array)
         {
            ArrayList ^ l_Vec = gcnew ArrayList();            
            IEnumerator ^ itr = obj->ArrayValue->GetEnumerator();
            while (itr->MoveNext())
            {
               l_Vec->Add(Encode((String^)itr->Current));
            }

            s->Append("(");
            s->Append(String::Join(",", reinterpret_cast<array<String^>^>(l_Vec->ToArray(String::typeid))));
            s->Append(")");
         }
         else if(obj->Type() == DataSet::DataSetType::Dictionary)
         {
            s->Append("{");
            IDictionaryEnumerator ^ itr = obj->DictionaryValue->GetEnumerator();
            while(itr->MoveNext())
            {
               s->Append(Encode((String^)itr->Key));
               s->Append("=");
               s->Append(Encode((String^)itr->Value));
               s->Append(";");
            }
            s->Append("}");
         }
         else if(obj->Type() == DataSet::DataSetType::String)
         {
            s->Append(Encode(obj->StringValue));
         }
         return s->ToString();
      }

      String ^ PrintWords(DataSet ^ Obj)
      {
         return Encode(Obj);
      }

      String ^ PrintWords(String ^ s)
      {
         DataSet ^ l_DataSet = gcnew DataSet(DataSet::DataSetType::String);
         l_DataSet->StringValue = s;
         return Encode(l_DataSet);
      }

      void CheckParam(DataSet ^ obj, DataSet::DataSetType l_type)
      {
         bool l_Valid = true;
         if (obj->Type() != l_type)
            l_Valid = false;
         switch (l_type)
         {
         case DataSet::DataSetType::Dictionary:
            if (!obj->DictionaryValue || (obj->DictionaryValue->Count == 0))
               l_Valid = false;
            break;
         case DataSet::DataSetType::Array:
            if (!obj->ArrayValue || (obj->ArrayValue->Count == 0))
               l_Valid = false;
            break;
         case DataSet::DataSetType::String:
            if (obj->StringValue->Length == 0)
               l_Valid = false;
            break;
         default:
            l_Valid = false;
         }
         if (!l_Valid)
            throw gcnew ProtocolException("invalid or missing argument");
      }

      void CheckParam(String ^ obj)
      {
         if (obj->Length == 0)
            throw gcnew ProtocolException("invalid or missing argument");
      }

      String ^ EncodeString(String ^ str)
      {
         return str->Replace("\\", "\\\\")
            ->Replace("\n", "\\e")
            ->Replace("\t", "\\t")
            ->Replace("\r", "")
            ->Replace("\"", "\\\"");         
      }

      String ^ DecodeString(String ^ str)
      {
         return str->Replace("\\e", "\n")
            ->Replace("\\t", "\t")
            ->Replace("\\\"", "\"")
            ->Replace("\\\\", "\\");         
      }

   public:

      CLI()
      {
      }

      ~CLI() 
      {
         Logout();
      }

      void Logout() 
      {
         try
         {
            if (m_Client)
            {
               StreamSend("Quit");
               ParseResponse();   
               m_Stream->Close();
               m_Client->Close();               
               m_Client = nullptr;               
            }
         }
         catch (...)
         {
         }
      }

      void Connect()
      {
         Connect(DefaultIp, DefaultPort, DefaultUser, DefaultPassword);
      }

      void Connect(String ^ ip, word port, String ^ user, String ^ password)
      {
         Connect(ip, port, user, password, DefaultSecureLogin);
      }

      void Connect(String ^ ip, word port, String ^ user, String ^ pass, bool in_SecureLogin)
      {
         m_Client = gcnew TcpClient(ip, port);
         m_Stream = m_Client->GetStream();

         String ^ prompt = Read();

         int l_Bracket = prompt->IndexOf('<');
         if (l_Bracket != -1)
         {
            m_Data = prompt->Substring(l_Bracket);
            l_Bracket = m_Data->LastIndexOf('>');
            if (l_Bracket != -1)
            {
              m_Data = m_Data->Substring(0, l_Bracket + 1);
            }
            else
            {
               throw gcnew InvalidServerPromptException(m_Data);               
            }
         }
         else
         {
            throw gcnew InvalidServerPromptException(m_Data);
         }

         if(in_SecureLogin)
         {
            m_Command = "APOP";

            MD5 ^ l_Md5 = gcnew MD5CryptoServiceProvider();         
            array<Byte> ^ l_Hash = l_Md5->ComputeHash(Encoding::ASCII->GetBytes(m_Data + pass));
            
            StringBuilder ^ l_AuthLogin = gcnew StringBuilder();
            l_AuthLogin->Append("APOP ");
            l_AuthLogin->Append(user);
            l_AuthLogin->Append(" ");

            for (int i = 0; i < 16; ++i)
            {
               Byte b1 = l_Hash[i] >> 4 & 0x0f;
               Byte b2 = l_Hash[i]      & 0x0f;
               l_AuthLogin->Append((wchar_t)(b1 + (b1 < 10 ? '0' : 'a' - 10)), 1);
               l_AuthLogin->Append((wchar_t)(b2 + (b2 < 10 ? '0' : 'a' - 10)), 1);
            }
            StreamSend(l_AuthLogin->ToString());

         }
         else
         {
            m_Command = "Login";
            StreamSend("user " + user);
            ParseResponse(false);
            if(m_ErrCode != 300)
            {
               String ^ l_ErrMsg = m_ErrMsg;
               word l_ErrCode = m_ErrCode;
               Logout();
               m_ErrMsg = l_ErrMsg;
               m_ErrCode = l_ErrCode;
               throw gcnew InvalidLoginException("Invalid username or password");
            }
            StreamSend("pass " + pass);
         }
         ParseResponse(false);
         if(m_ErrCode != 200)
         {
            String ^ l_ErrMsg = m_ErrMsg;
            word l_ErrCode = m_ErrCode;
            Logout();
            m_ErrMsg = l_ErrMsg;
            m_ErrCode = l_ErrCode;
            throw gcnew InvalidLoginException("Invalid username or password");
         }
         StreamSend("inline");
         ParseResponse(false);
      }

      void SendCommand(String ^ cmd)
      {
         m_Command = "SendCommand";
         StreamSend(cmd);
         ParseResponse();
      }

      DataSet ^ GetResponseData()
      {
         return ParseWords(GetWords());
      }

      DataSet ^ ListAccounts(String ^ domainName)
      {
         m_Command = "ListAccounts";
         if(domainName->Length)
            m_Command += " " + domainName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void CreateAccount(String ^ accountName, DataSet ^ settings, String ^ accountType, bool external)
      {
         m_Command = "CreateAccount";
         CheckParam(accountName);
         m_Command += " " + accountName;
         if(accountType->Length)
            m_Command += " " + accountType;
         if(external)
            m_Command += " external";
         if(settings->DictionaryValue->Count)
            StreamSend(m_Command + " " + PrintWords(settings));
         else
            StreamSend(m_Command);
         ParseResponse();
      }

      void RenameAccount(String ^ oldAccountName, String ^ newAccountName)
      {
         m_Command = "RenameAccount";
         CheckParam(oldAccountName);
         CheckParam(newAccountName);
         m_Command += " " + oldAccountName + " into " + newAccountName;
         StreamSend(m_Command);
         ParseResponse();
      }

      void DeleteAccount(String ^ accountName)
      {
         m_Command = "DeleteAccount";
         CheckParam(accountName);
         m_Command += " " + accountName;
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ GetAccountSettings(String ^ accountName)
      {
         m_Command = "GetAccountSettings";
         CheckParam(accountName);
         m_Command += " " + accountName;
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      DataSet ^ GetAccount(String ^ accountName)
      {
         return GetAccountSettings(accountName);
      }

      DataSet ^ GetAccountEffectiveSettings(String ^ accountName)
      {
         m_Command = "getAccountEffectiveSettings";
         CheckParam(accountName);
         m_Command += " " + accountName;
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void UpdateAccountSettings(String ^ accountName, DataSet ^ settings)
      {
         m_Command = "UpdateAccountSettings";
         CheckParam(accountName);
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + accountName + " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      void UpdateAccount(String ^ accountName, DataSet ^ settings)
      {
         UpdateAccountSettings(accountName, settings);
      }

      void SetAccountSettings(String ^ accountName, DataSet ^ settings)
      {
         m_Command = "SetAccountSettings";
         CheckParam(accountName);
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + accountName + " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      void SetAccount(String ^ accountName, DataSet ^ settings)
      {
         SetAccountSettings(accountName, settings);
      }

      void SetAccountPassword(String ^ accountName, String ^ newPassword)
      {
         m_Command = "SetAccountPassword";
         CheckParam(accountName);
         CheckParam(newPassword);
         m_Command += " " + accountName + " TO " + PrintWords(newPassword);
         SendCmd(m_Command);
      }

      bool VerifyAccountPassword(String ^ accountName, String ^ password)
      {
         m_Command = "VerifyAccountPassword";
         CheckParam(accountName);
         CheckParam(password);
         m_Command += " " + accountName + " PASSWORD " + PrintWords(password);
         try
         {
            SendCmd(m_Command);
            return true;
         }
         catch(Exception ^)
         {
            if(m_ErrCode != 515)
               throw;
            return false;
         }
      }

      DataSet ^ GetAccountAliases(String ^ accountName)
      {
         m_Command = "GetAccountAliases";
         CheckParam(accountName);
         m_Command += " " + accountName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void SetAccountAliases(String ^ accountName, DataSet ^ aliases)
      {
         m_Command = "SetAccountAliases";
         CheckParam(accountName);
         CheckParam(aliases, DataSet::DataSetType::Array);
         m_Command += " " + accountName + " " + PrintWords(aliases);
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ GetAccountRules(String ^ accountName)
      {
         m_Command = "GetAccountRules";
         CheckParam(accountName);
         m_Command += " " + accountName;
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetAccountRules(String ^ accountName, DataSet ^ rules)
      {
         m_Command = "SetAccountRules";
         CheckParam(accountName);
         CheckParam(rules, DataSet::DataSetType::Array);
         m_Command += " " + accountName + " " + PrintWords(rules);
         SendCmd(m_Command);
      }

      DataSet ^ GetAccountRPOP(String ^ accountName)
      {
         m_Command = "GetAccountRPOP";
         CheckParam(accountName);
         m_Command += " " + accountName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void SetAccountRPOP(String ^ accountName, DataSet ^ records)
      {
         m_Command = "SetAccountRPOP";
         CheckParam(accountName);
         CheckParam(records, DataSet::DataSetType::Array);
         m_Command += " " + accountName + " " + PrintWords(records);
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ GetAccountRights(String ^ accountName)
      {
         m_Command = "GetAccountRights";
         CheckParam(accountName);
         m_Command += " " + accountName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void SetAccountRights(String ^ accountName, DataSet ^ rights)
      {
         m_Command = "SetAccountRights";
         CheckParam(accountName);
         CheckParam(rights, DataSet::DataSetType::Array);
         m_Command += " " + accountName + " " + PrintWords(rights);
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ GetAccountInfo(String ^ accountName, String ^ key)
      {
         m_Command = "GetAccountInfo";
         CheckParam(accountName);
         CheckParam(key);
         m_Command += " " + accountName + " Key " + key;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      DataSet ^ GetWebUser(String ^ accountName)
      {
         m_Command = "GetWebUser";
         CheckParam(accountName);
         m_Command += " " + accountName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void SetWebUser(String ^ accountName, DataSet ^ settings)
      {
         m_Command = "SetWebUser";
         CheckParam(accountName);
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + accountName + " " + PrintWords(settings);
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ GetEffectiveWebUser(String ^ accountName)
      {
         m_Command = "GetEffectiveWebUser";
         CheckParam(accountName);
         m_Command += " " + accountName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      DataSet ^ ListGroups(String ^ domainName)
      {
         m_Command = "ListGroups";
         if(domainName->Length)
            m_Command += " " + domainName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void CreateGroup(String ^ groupName, DataSet ^ settings)
      {
         m_Command = "CreateGroup";
         CheckParam(groupName);
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + groupName + " " + PrintWords(settings);
         StreamSend(m_Command);
         ParseResponse();
      }

      void RenameGroup(String ^ oldGroupName, String ^ newGroupName)
      {
         m_Command = "RenameGroup";
         CheckParam(oldGroupName);
         CheckParam(newGroupName);
         m_Command += " " + oldGroupName + " into " + newGroupName;
         StreamSend(m_Command);
         ParseResponse();
      }

      void DeleteGroup(String ^ groupName)
      {
         m_Command = "DeleteGroup";
         CheckParam(groupName);
         m_Command += " " + groupName;
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ GetGroup(String ^ groupName)
      {
         m_Command = "GetGroup";
         CheckParam(groupName);
         m_Command += " " + groupName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void SetGroup(String ^ groupName, DataSet ^ settings)
      {
         m_Command = "SetGroup";
         CheckParam(groupName);
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + groupName + " " + PrintWords(settings);
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ ListForwarders(String ^ domainName)
      {
         m_Command = "ListForwarders";
         if(domainName->Length)
            m_Command += " " + domainName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void CreateForwarder(String ^ forwarderName, String ^ address)
      {
         m_Command = "CreateForwarder";
         CheckParam(forwarderName);
         CheckParam(address);
         m_Command += " " + forwarderName + " to " + address;
         StreamSend(m_Command);
         ParseResponse();
      }

      void DeleteForwarder(String ^ forwarderName)
      {
         m_Command = "DeleteForwarder";
         CheckParam(forwarderName);
         m_Command += " " + forwarderName;
         StreamSend(m_Command);
         ParseResponse();
      }

      String ^ GetForwarder(String ^ forwarderName)
      {
         m_Command = "GetForwarder";
         CheckParam(forwarderName);
         m_Command += " " + forwarderName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords())->StringValue;
      }

      DataSet ^ ListDomains()
      {
         m_Command = "ListDomains";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      String ^ MainDomainName()
      {
         m_Command = "MainDomainName";
         SendCmd(m_Command);
         return ParseWords(GetWords())->StringValue;
      }

      DataSet ^ GetDomainSettings(String ^ domainName)
      {
         m_Command = "getDomainSettings";
         if(domainName->Length)
            m_Command += " " + domainName;
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      DataSet ^ GetDomain(String ^ domainName)
      {
         return GetDomainSettings(domainName);
      }

      DataSet ^ GetDomainEffectiveSettings(String ^ domainName)
      {
         m_Command = "getDomainEffectiveSettings";
         if(domainName->Length)
            m_Command += " " + domainName;
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void UpdateDomainSettings(String ^ domainName, DataSet ^ settings)
      {
         m_Command = "UpdateDomainSettings";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         if(domainName->Length)
            m_Command += " " + domainName;
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      void UpdateDomain(String ^ domainName, DataSet ^ settings)
      {
         UpdateDomainSettings(domainName, settings);
      }

      void SetDomainSettings(String ^ domainName, DataSet ^ settings)
      {
         m_Command = "SetDomainSettings";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         if(domainName->Length)
            m_Command += " " + domainName;
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      void SetDomain(String ^ domainName, DataSet ^ settings)
      {
         SetDomainSettings(domainName, settings);
      }

      void CreateDomain(String ^ domainName, DataSet ^ settings)
      {
         m_Command = "CreateDomain";
         CheckParam(domainName);
         m_Command += " " + domainName;
         if(settings->DictionaryValue->Count)
            m_Command += " " + PrintWords(settings);
         StreamSend(m_Command);
         ParseResponse();
      }

      void CreateSharedDomain(String ^ domainName, DataSet ^ settings)
      {
         m_Command = "CreateSharedDomain";
         CheckParam(domainName);
         m_Command += " " + domainName;
         if(settings->DictionaryValue->Count)
            m_Command += " " + PrintWords(settings);
         StreamSend(m_Command);
         ParseResponse();
      }

      void CreateDirectoryDomain(String ^ domainName, DataSet ^ settings)
      {
         m_Command = "CreateDirectoryDomain";
         CheckParam(domainName);
         m_Command += " " + domainName;
         if(settings->DictionaryValue->Count)
            m_Command += " " + PrintWords(settings);
         StreamSend(m_Command);
         ParseResponse();
      }

      void ReloadDirectoryDomains()
      {
         m_Command = "ReloadDirectoryDomains";
         StreamSend(m_Command);
         ParseResponse();
      }

      void RenameDomain(String ^ oldDomainName, String ^ newDomainName)
      {
         m_Command = "RenameDomain";
         CheckParam(oldDomainName);
         CheckParam(newDomainName);
         m_Command += " " + oldDomainName + " into " + newDomainName;
         StreamSend(m_Command);
         ParseResponse();
      }

      void DeleteDomain(String ^ domainName, bool force)
      {
         m_Command = "DeleteDomain";
         CheckParam(domainName);
         m_Command += " " + domainName;
         if(force)
            m_Command += " force";
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ GetDomainRules(String ^ domainName)
      {
         m_Command = "GetDomainRules";
         CheckParam(domainName);
         m_Command += " " + domainName;
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetDomainRules(String ^ domainName, DataSet ^ rules)
      {
         m_Command = "SetDomainRules";
         CheckParam(domainName);
         CheckParam(rules, DataSet::DataSetType::Array);
         m_Command += " " + domainName + " " + PrintWords(rules);
         SendCmd(m_Command);
      }

      DataSet ^ GetDomainAliases(String ^ domainName)
      {
         m_Command = "GetDomainAliases";
         if(domainName->Length)
            m_Command += " " + domainName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void SetDomainAliases(String ^ domainName, DataSet ^ aliases)
      {
         m_Command = "SetDomainAliases";
         CheckParam(aliases, DataSet::DataSetType::Array);
         if(domainName->Length)
            m_Command += " " + domainName;
         m_Command += " " + PrintWords(aliases);
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ ListAdminDomains(String ^ domainName)
      {
         m_Command = "ListAdminDomains";
         if(domainName->Length)
            m_Command += " " + domainName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void InsertDirectoryRecords(String ^ domainName)
      {
         m_Command = "insertDirectoryRecords";
         if(domainName->Length)
            m_Command += " " + domainName;
         StreamSend(m_Command);
         ParseResponse();
      }

      void DeleteDirectoryRecords(String ^ domainName)
      {
         m_Command = "deleteDirectoryRecords";
         if(domainName->Length)
            m_Command += " " + domainName;
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ GetDirectoryIntegration()
      {
         m_Command = "GetDirectoryIntegration";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetDirectoryIntegration(DataSet ^ settings)
      {
         m_Command = "SetDirectoryIntegration";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      DataSet ^ GetClusterDirectoryIntegration()
      {
         m_Command = "GetClusterDirectoryIntegration";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetClusterDirectoryIntegration(DataSet ^ settings)
      {
         m_Command = "SetClusterDirectoryIntegration";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      DataSet ^ GetDomainDefaults()
      {
         m_Command = "GetDomainDefaults";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void UpdateDomainDefaults(DataSet ^ settings)
      {
         m_Command = "UpdateDomainDefaults";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      void SetDomainDefaults(DataSet ^ settings)
      {
         m_Command = "SetDomainDefaults";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      DataSet ^ GetClusterDomainDefaults()
      {
         m_Command = "GetClusterDomainDefaults";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void UpdateClusterDomainDefaults(DataSet ^ settings)
      {
         m_Command = "UpdateClusterDomainDefaults";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      void SetClusterDomainDefaults(DataSet ^ settings)
      {
         m_Command = "SetClusterDomainDefaults";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      DataSet ^ GetAllAccountsDefaults()
      {
         m_Command = "GetAllAccountsDefaults";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void UpdateAllAccountsDefaults(DataSet ^ settings)
      {
         m_Command = "UpdateAllAccountsDefaults";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      void SetAllAccountsDefaults(DataSet ^ settings)
      {
         m_Command = "SetAllAccountsDefaults";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      DataSet ^ GetClusterAccountDefaults()
      {
         m_Command = "GetClusterAccountDefaults";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void UpdateClusterAccountDefaults(DataSet ^ settings)
      {
         m_Command = "UpdateClusterAccountDefaults";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      void SetClusterAccountDefaults(DataSet ^ settings)
      {
         m_Command = "SetClusterAccountDefaults";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      DataSet ^ GetServerWebUserDefaults()
      {
         m_Command = "GetServerWebUserDefaults";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetServerWebUserDefaults(DataSet ^ settings)
      {
         m_Command = "SetServerWebUserDefaults";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      DataSet ^ GetClusterWebUserDefaults()
      {
         m_Command = "GetClusterWebUserDefaults";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetClusterWebUserDefaults(DataSet ^ settings)
      {
         m_Command = "SetClusterWebUserDefaults";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      String ^ GetDomainLocation(String ^ domainName)
      {
         m_Command = "GetDomainLocation";
         CheckParam(domainName);
         m_Command += " " + domainName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords())->StringValue;
      }

      String ^ GetAccountLocation(String ^ accountName)
      {
         m_Command = "GetAccountLocation";
         CheckParam(accountName);
         m_Command += " " + accountName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords())->StringValue;
      }

      DataSet ^ GetAccountDefaults(String ^ domainName)
      {
         m_Command = "GetAccountDefaults";
         if(domainName->Length)
            m_Command += " " + domainName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void UpdateAccountDefaults(String ^ domainName, DataSet ^ settings)
      {
         m_Command = "UpdateAccountDefaults";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         if(domainName->Length)
            m_Command += " " + domainName;
         m_Command += " " + PrintWords(settings);
         StreamSend(m_Command);
         ParseResponse();
      }

      void SetAccountDefaults(String ^ domainName, DataSet ^ settings)
      {
         m_Command = "SetAccountDefaults";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         if(domainName->Length)
            m_Command += " " + domainName;
         m_Command += " " + PrintWords(settings);
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ GetWebUserDefaults(String ^ domainName)
      {
         m_Command = "GetWebUserDefaults";
         if(domainName->Length)
            m_Command += " " + domainName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void SetWebUserDefaults(String ^ domainName, DataSet ^ settings)
      {
         m_Command = "SetWebUserDefaults";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         if(domainName->Length)
            m_Command += " " + domainName;
         m_Command += " " + PrintWords(settings);
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ GetAccountTemplate(String ^ domainName)
      {
         m_Command = "GetAccountTemplate";
         if(domainName->Length)
            m_Command += " " + domainName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void UpdateAccountTemplate(String ^ domainName, DataSet ^ settings)
      {
         m_Command = "UpdateAccountTemplate";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         if(domainName->Length)
            m_Command += " " + domainName;
         m_Command += " " + PrintWords(settings);
         StreamSend(m_Command);
         ParseResponse();
      }

      void SetAccountTemplate(String ^ domainName, DataSet ^ settings)
      {
         m_Command = "SetAccountTemplate";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         if(domainName->Length)
            m_Command += " " + domainName;
         m_Command += " " + PrintWords(settings);
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ ListMailboxes(String ^ accountName, String ^ filter, String ^ auth)
      {
         m_Command = "ListMailboxes";
         CheckParam(accountName);
         m_Command += " " + accountName;
         if(filter->Length)
            m_Command += " FILTER \"" + filter + "\"";
         if(auth->Length)
            m_Command += " AUTH " + auth;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void CreateMailbox(String ^ accountName, String ^ mailboxName, String ^ auth)
      {
         m_Command = "CreateMailbox";
         CheckParam(accountName);
         CheckParam(mailboxName);
         m_Command += " " + accountName + " MAILBOX \"" + mailboxName + "\"";
         if(auth->Length)
            m_Command += " AUTH " + auth;
         StreamSend(m_Command);
         ParseResponse();
      }

      void RenameMailbox(String ^ accountName, String ^ oldMailboxName, String ^ newMailboxName, String ^ auth)
      {
         m_Command = "RenameMailbox";
         CheckParam(accountName);
         CheckParam(oldMailboxName);
         CheckParam(newMailboxName);
         m_Command += " " + accountName + " MAILBOX \"" + oldMailboxName + "\" INTO \"" + newMailboxName + "\"";
         if(auth->Length)
            m_Command += " AUTH " + auth;
         StreamSend(m_Command);
         ParseResponse();
      }

      void RenameMailboxes(String ^ accountName, String ^ oldMailboxName, String ^ newMailboxName, String ^ auth)
      {
         m_Command = "RenameMailbox";
         CheckParam(accountName);
         CheckParam(oldMailboxName);
         CheckParam(newMailboxName);
         m_Command += " " + accountName + " MAILBOXES \"" + oldMailboxName + "\" INTO \"" + newMailboxName + "\"";
         if(auth->Length)
            m_Command += " AUTH " + auth;
         StreamSend(m_Command);
         ParseResponse();
      }

      void DeleteMailbox(String ^ accountName, String ^ mailboxName, String ^ auth)
      {
         m_Command = "DeleteMailbox";
         CheckParam(accountName);
         CheckParam(mailboxName);
         m_Command += " " + accountName + " MAILBOX \"" + mailboxName + "\"";
         if(auth->Length)
            m_Command += " AUTH " + auth;
         StreamSend(m_Command);
         ParseResponse();
      }

      void DeleteMailboxes(String ^ accountName, String ^ mailboxName, String ^ auth)
      {
         m_Command = "DeleteMailbox";
         CheckParam(accountName);
         CheckParam(mailboxName);
         m_Command += " " + accountName + " MAILBOXES \"" + mailboxName + "\"";
         if(auth->Length)
            m_Command += " AUTH " + auth;
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ GetMailboxInfo(String ^ accountName, String ^ mailboxName, String ^ auth)
      {
         m_Command = "GetMailboxInfo";
         CheckParam(accountName);
         CheckParam(mailboxName);
         m_Command += " " + accountName + " MAILBOX \"" + mailboxName + "\"";
         if(auth->Length)
            m_Command += " AUTH " + auth;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      DataSet ^ GetMailboxACL(String ^ accountName, String ^ mailboxName, String ^ auth)
      {
         m_Command = "GetMailboxACL";
         CheckParam(accountName);
         CheckParam(mailboxName);
         m_Command += " " + accountName + " MAILBOX \"" + mailboxName + "\"";
         if(auth->Length)
            m_Command += " AUTH " + auth;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void SetMailboxACL(String ^ accountName, String ^ mailboxName, DataSet ^ newACL, String ^ auth)
      {
         m_Command = "SetMailboxACL";
         CheckParam(accountName);
         CheckParam(mailboxName);
         CheckParam(newACL, DataSet::DataSetType::Dictionary);
         m_Command += " " + accountName + " MAILBOX \"" + mailboxName + "\"";
         if(auth->Length)
            m_Command += " AUTH " + auth;
         m_Command += " " + PrintWords(newACL);
         StreamSend(m_Command);
         ParseResponse();
      }

      void ResetMailboxACL(String ^ accountName, String ^ mailboxName, DataSet ^ newACL, String ^ auth)
      {
         DataSet ^ l_MailboxACL = GetMailboxACL(accountName, mailboxName, auth);      
         ArrayList ^ Keys = gcnew ArrayList();
         IDictionaryEnumerator ^ itr = l_MailboxACL->DictionaryValue->GetEnumerator();
         while (itr->MoveNext())
         {
            Keys->Add(itr->Key);            
         }
         IEnumerator ^ en = Keys->GetEnumerator();
         while (en->MoveNext())
         {
            ((DataSet ^)(l_MailboxACL->DictionaryValue->Item[(String ^)en->Current]))->StringValue = "";
         }
         itr = newACL->DictionaryValue->GetEnumerator();
         while (itr->MoveNext())
         {
            l_MailboxACL->DictionaryValue->Item[(String^)itr->Key] = (DataSet^)itr->Value;
         }
         SetMailboxACL(accountName, mailboxName, l_MailboxACL, auth);
      }

      String ^ GetMailboxRights(String ^ accountName, String ^ mailboxName, String ^ auth)
      {
         m_Command = "GetMailboxRights";
         CheckParam(accountName);
         CheckParam(mailboxName);
         CheckParam(auth);
         m_Command += " " + accountName + " MAILBOX \"" + mailboxName + "\" AUTH " + auth;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords())->StringValue;
      }

      void SetMailboxClass(String ^ accountName, String ^ mailboxName, String ^ newClass, String ^ auth)
      {
         m_Command = "SetMailboxClass";
         CheckParam(accountName);
         CheckParam(mailboxName);
         CheckParam(newClass);
         m_Command += " " + accountName + " MAILBOX \"" + mailboxName + "\"";
         if(auth->Length)
            m_Command += " AUTH " + auth;
         m_Command += " CLASS " + PrintWords(newClass);
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ GetAccountSubscription(String ^ accountName)
      {
         m_Command = "GetAccountSubscription";
         CheckParam(accountName);
         m_Command += " " + accountName;
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void AddAccountSubscription(String ^ accountName, DataSet ^ subscription)
      {
         DataSet ^ l_Subscription = GetAccountSubscription(accountName);   
         if (subscription->Type() == DataSet::DataSetType::String)
         {
            IEnumerator ^itr = l_Subscription->ArrayValue->GetEnumerator();
            while(itr->MoveNext())
            {
               if (((DataSet^)itr->Current)->StringValue == subscription->StringValue)
               {
                  return;
               }         
            }
            l_Subscription->ArrayValue->Add(subscription);
         }
         else
         {
            IEnumerator ^itr = subscription->ArrayValue->GetEnumerator();
            while(itr->MoveNext())
            {
               l_Subscription->ArrayValue->Add(DataSet::String(((DataSet^)itr->Current)->StringValue));
            }
         }
         SetAccountSubscription(accountName, l_Subscription);
      }

      void SetAccountSubscription(String ^ accountName, DataSet ^ subscription)
      {
         m_Command = "SetAccountSubscription";
         CheckParam(accountName);
         CheckParam(subscription, DataSet::DataSetType::Array);
         m_Command += " " + accountName + " " + PrintWords(subscription);
         SendCmd(m_Command);
      }

      DataSet ^ GetMailboxAliases(String ^ accountName)
      {
         m_Command = "GetMailboxAliases";
         CheckParam(accountName);
         m_Command += " " + accountName;
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetMailboxAliases(String ^ accountName, DataSet ^ aliases)
      {
         m_Command = "setMailboxAliases";
         CheckParam(accountName);
         CheckParam(aliases, DataSet::DataSetType::Dictionary);
         m_Command += " " + accountName + " " + PrintWords(aliases);
         SendCmd(m_Command);
      }

      DataSet ^ GetDomainAlerts(String ^ domainName)
      {
         m_Command = "GetDomainAlerts";
         if(domainName->Length)
            m_Command += " " + domainName;
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetDomainAlerts(String ^ domainName, DataSet ^ alerts)
      {
         m_Command = "SetDomainAlerts";
         CheckParam(alerts, DataSet::DataSetType::Dictionary);
         if(domainName->Length)
            m_Command += " " + domainName;
         m_Command += " " + PrintWords(alerts);
         SendCmd(m_Command);
      }

      void PostDomainAlert(String ^ domainName, String ^ alert)
      {
         m_Command = "PostDomainAlert";
         CheckParam(domainName);
         CheckParam(alert);
         m_Command += " " + domainName + " ALERT " + PrintWords(alert);
         SendCmd(m_Command);
      }

      void RemoveDomainAlert(String ^ domainName, String ^ timeStamp)
      {
         m_Command = "RemoveDomainAlert";
         CheckParam(domainName);
         CheckParam(timeStamp);
         m_Command += " " + domainName + " ALERT " + timeStamp;
         SendCmd(m_Command);
      }

      DataSet ^ GetAccountAlerts(String ^ accountName)
      {
         m_Command = "GetAccountAlerts";
         CheckParam(accountName);
         m_Command += " " + accountName;
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetAccountAlerts(String ^ accountName, DataSet ^ alerts)
      {
         m_Command = "SetAccountAlerts";
         CheckParam(accountName);
         CheckParam(alerts, DataSet::DataSetType::Dictionary);
         m_Command += " " + accountName + " " + PrintWords(alerts);
         SendCmd(m_Command);
      }

      void PostAccountAlert(String ^ accountName, String ^ alert)
      {
         m_Command = "PostAccountAlert";
         CheckParam(accountName);
         CheckParam(alert);
         m_Command += " " + accountName + " ALERT " + PrintWords(alert);
         SendCmd(m_Command);
      }

      void RemoveAccountAlert(String ^ accountName, String ^ timeStamp)
      {
         m_Command = "RemoveAccountAlert";
         CheckParam(accountName);
         CheckParam(timeStamp);
         m_Command += " " + accountName + " ALERT " + timeStamp;
         SendCmd(m_Command);
      }

      DataSet ^ GetServerAlerts()
      {
         m_Command = "GetServerAlerts";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetServerAlerts(DataSet ^ alerts)
      {
         m_Command = "SetServerAlerts";
         CheckParam(alerts, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(alerts);
         SendCmd(m_Command);
      }

      void PostServerAlert(String ^ alert)
      {
         m_Command = "PostServerAlert";
         CheckParam(alert);
         m_Command += " " + PrintWords(alert);
         SendCmd(m_Command);
      }

      void RemoveServerAlert(String ^ timeStamp)
      {
         m_Command = "RemoveServerAlert";
         CheckParam(timeStamp);
         m_Command += " " + timeStamp;
         SendCmd(m_Command);
      }

      DataSet ^ GetClusterAlerts()
      {
         m_Command = "GetClusterAlerts";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetClusterAlerts(DataSet ^ alerts)
      {
         m_Command = "SetClusterAlerts";
         CheckParam(alerts, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(alerts);
         SendCmd(m_Command);
      }

      void PostClusterAlert(String ^ alert)
      {
         m_Command = "PostClusterAlert";
         CheckParam(alert);
         m_Command += " " + PrintWords(alert);
         SendCmd(m_Command);
      }

      void RemoveClusterAlert(String ^ timeStamp)
      {
         m_Command = "RemoveClusterAlert";
         CheckParam(timeStamp);
         m_Command += " " + timeStamp;
         SendCmd(m_Command);
      }

      DataSet ^ GetWebFile(String ^ accountName, String ^ fileName)
      {
         m_Command = "GetWebFile";
         CheckParam(accountName);
         CheckParam(fileName);
         m_Command += " " + accountName + " FILE \"" + fileName + "\"";
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      DataSet ^ GetWebFile(String ^ accountName, String ^ fileName, int position, int sliceSize)
      {
         m_Command = "GetWebFile";
         CheckParam(accountName);
         CheckParam(fileName);
         m_Command += " " + accountName + " FILE \"" + fileName + "\"";
         m_Command += " OFFSET " + position.ToString() + " SIZE " + sliceSize.ToString();
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void PutWebFile(String ^ accountName, String ^ fileName, String ^ in_data)
      {
         m_Command = "PutWebFile";
         CheckParam(accountName);
         CheckParam(fileName);
         CheckParam(in_data);
         m_Command += " " + accountName + " FILE \"" + fileName + "\" DATA " + in_data;
         StreamSend(m_Command);
         ParseResponse();
      }

      void PutWebFile(String ^ accountName, String ^ fileName, String ^ in_data, int position)
      {
         m_Command = "PutWebFile";
         CheckParam(accountName);
         CheckParam(fileName);
         CheckParam(in_data);
         m_Command += " " + accountName + " FILE \"" + fileName + "\" OFFSET " + position.ToString() + " DATA " + in_data;
         StreamSend(m_Command);
         ParseResponse();
      }

      void RenameWebFile(String ^ accountName, String ^ fileName, String ^ newFileName)
      {
         m_Command = "RenameWebFile";
         CheckParam(accountName);
         CheckParam(fileName);
         CheckParam(newFileName);
         m_Command += " " + accountName + " FILE \"" + fileName + "\" INTO \"" + newFileName + "\"";
         StreamSend(m_Command);
         ParseResponse();
      }

      void DeleteWebFile(String ^ accountName, String ^ fileName)
      {
         m_Command = "DeleteWebFile";
         CheckParam(accountName);
         CheckParam(fileName);
         m_Command += " " + accountName + " FILE \"" + fileName + "\"";
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ ListWebFiles(String ^ accountName, String ^ path)
      {
         m_Command = "ListWebFiles";
         CheckParam(accountName);
         m_Command += " " + accountName;
         if(path->Length)
            m_Command += " PATH \"" + path + "\"";
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      DataSet ^ GetWebFilesInfo(String ^ accountName)
      {
         m_Command = "GetWebFilesInfo";
         CheckParam(accountName);
         m_Command += " " + accountName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      DataSet ^ ListLists(String ^ domainName)
      {
         m_Command = "ListLists";
         if(domainName->Length)
            m_Command += " " + domainName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      DataSet ^ GetDomainLists(String ^ domainName)
      {
         m_Command = "GetDomainLists";
         if(domainName->Length)
            m_Command += " " + domainName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      DataSet ^ GetAccountLists(String ^ accountName)
      {
         m_Command = "GetAccountLists";
         CheckParam(accountName);
         m_Command += " " + accountName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void CreateList(String ^ listName, String ^ accountName)
      {
         m_Command = "CreateList";
         CheckParam(listName);
         CheckParam(accountName);
         m_Command += " " + listName + " for " + accountName;
         StreamSend(m_Command);
         ParseResponse();
      }

      void RenameList(String ^ listName, String ^ newListName)
      {
         m_Command = "RenameList";
         CheckParam(listName);
         CheckParam(newListName);
         m_Command += " " + listName + " into " + newListName;
         StreamSend(m_Command);
         ParseResponse();
      }

      void DeleteList(String ^ listName)
      {
         m_Command = "deleteList";
         CheckParam(listName);
         m_Command += " " + listName;
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ GetList(String ^ listName)
      {
         m_Command = "GetList";
         CheckParam(listName);
         m_Command += " " + listName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void UpdateList(String ^ listName, DataSet ^ settings)
      {
         m_Command = "UpdateList";
         CheckParam(listName);
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + listName + " " + PrintWords(settings);
         StreamSend(m_Command);
         ParseResponse();
      }

      void List(String ^ listName, String ^ operation, String ^ subscriberAddress, bool silently, bool confirm)
      {
         m_Command = "List";
         CheckParam(listName);
         CheckParam(operation);
         CheckParam(subscriberAddress);
         m_Command += " " + listName + " " + operation;
         if(silently)
            m_Command += " silently";
         if(confirm)
            m_Command += " confirm";
         m_Command += " " + subscriberAddress;
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ ListSubscribers(String ^ listName, String ^ filter, int limit)
      {
         m_Command = "ListSubscribers";
         CheckParam(listName);
         m_Command += " " + listName;
         if(filter->Length || limit > 0)
            m_Command += " FILTER \"" + (filter->Length ? filter : "") + "\"";
         if(limit > 0)
            m_Command += " " + limit;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      DataSet ^ GetSubscriberInfo(String ^ listName, String ^ subscriberAddress)
      {
         m_Command = "GetSubscriberInfo";
         CheckParam(listName);
         CheckParam(subscriberAddress);
         m_Command += " " + listName + " NAME " + subscriberAddress;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void SetPostingMode(String ^ listName, String ^ subscriberAddress, String ^ mode)
      {
         m_Command = "SetPostingMode";
         CheckParam(listName);
         CheckParam(subscriberAddress);
         CheckParam(mode);
         m_Command += " " + listName + " FOR " + subscriberAddress + " " + mode;
         StreamSend(m_Command);
         ParseResponse();
      }

      void ProcessBounce(String ^ listName, String ^ subscriberAddress, bool fatal)
      {
         m_Command = "ProcessBounce";
         CheckParam(listName);
         CheckParam(subscriberAddress);
         m_Command += " " + listName;
         if(fatal)
            m_Command += " FATAL";
         m_Command += " FOR " + subscriberAddress;
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ ListDomainSkins(String ^ domainName)
      {
         m_Command = "ListDomainSkins";
         if(domainName->Length)
            m_Command += " " + domainName;
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void CreateDomainSkin(String ^ domainName, String ^ skinName)
      {
         m_Command = "CreateDomainSkin";
         CheckParam(skinName);
         if(domainName->Length)
            m_Command += " " + domainName + " SKIN";
         m_Command += " \"" + skinName + "\"";
         SendCmd(m_Command);
      }

      void RenameDomainSkin(String ^ domainName, String ^ skinName, String ^ newName)
      {
         m_Command = "RenameDomainSkin";
         CheckParam(skinName);
         CheckParam(newName);
         if(domainName->Length)
            m_Command += " " + domainName + " SKIN";
         m_Command += " \"" + skinName + "\" INTO \"" + newName + "\"";
         SendCmd(m_Command);
      }

      void DeleteDomainSkin(String ^ domainName, String ^ skinName)
      {
         m_Command = "DeleteDomainSkin";
         CheckParam(skinName);
         if(domainName->Length)
            m_Command += " " + domainName + " SKIN";
         m_Command += " \"" + skinName + "\"";
         SendCmd(m_Command);
      }

      DataSet ^ ListDomainSkinFiles(String ^ domainName, String ^ skinName)
      {
         m_Command = "ListDomainSkinFiles";
         CheckParam(skinName);
         if(domainName->Length)
            m_Command += " " + domainName + " SKIN";
         m_Command += " \"" + skinName + "\"";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      DataSet ^ ReadDomainSkinFile(String ^ domainName, String ^ skinName, String ^ fileName)
      {
         m_Command = "ReadDomainSkinFile";
         CheckParam(skinName);
         CheckParam(fileName);
         if(domainName->Length)
            m_Command += " " + domainName + " SKIN";
         m_Command += " \"" + skinName + "\" FILE \"" + fileName + "\"";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void StoreDomainSkinFile(String ^ domainName, String ^ skinName, String ^ fileName, String ^ in_data)
      {
         m_Command = "StoreDomainSkinFile";
         CheckParam(skinName);
         CheckParam(fileName);
         CheckParam(in_data);
         if(domainName->Length)
            m_Command += " " + domainName + " SKIN";
         m_Command += " \"" + skinName + "\" FILE \"" + fileName + "\" DATA \"" + in_data + "\"";
         SendCmd(m_Command);
      }

      void DeleteDomainSkinFile(String ^ domainName, String ^ skinName, String ^ fileName)
      {
         m_Command = "StoreDomainSkinFile";
         CheckParam(skinName);
         CheckParam(fileName);
         if(domainName->Length)
            m_Command += " " + domainName + " SKIN";
         m_Command += " \"" + skinName + "\" FILE \"" + fileName + "\" DELETE";
         SendCmd(m_Command);
      }

      DataSet ^ ListServerSkins()
      {
         m_Command = "ListServerSkins";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void CreateServerSkin(String ^ skinName)
      {
         m_Command = "CreateServerSkin";
         CheckParam(skinName);
         m_Command += " \"" + skinName + "\"";
         SendCmd(m_Command);
      }

      void RenameServerSkin(String ^ skinName, String ^ newName)
      {
         m_Command = "RenameServerSkin";
         CheckParam(skinName);
         CheckParam(newName);
         m_Command += " \"" + skinName + "\" INTO \"" + newName + "\"";
         SendCmd(m_Command);
      }

      void DeleteServerSkin(String ^ skinName)
      {
         m_Command = "DeleteServerSkin";
         CheckParam(skinName);
         m_Command += " \"" + skinName + "\"";
         SendCmd(m_Command);
      }

      DataSet ^ ListServerSkinFiles(String ^ skinName)
      {
         m_Command = "ListServerSkinFiles";
         CheckParam(skinName);
         m_Command += " \"" + skinName + "\"";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      DataSet ^ ReadServerSkinFile(String ^ skinName, String ^ fileName)
      {
         m_Command = "ReadServerSkinFile";
         CheckParam(skinName);
         CheckParam(fileName);
         m_Command += " \"" + skinName + "\" FILE \"" + fileName + "\"";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void StoreServerSkinFile(String ^ skinName, String ^ fileName, String ^ in_data)
      {
         m_Command = "StoreServerSkinFile";
         CheckParam(skinName);
         CheckParam(fileName);
         CheckParam(in_data);
         m_Command += " \"" + skinName + "\" FILE \"" + fileName + "\" DATA \"" + in_data + "\"";
         SendCmd(m_Command);
      }

      void DeleteServerSkinFile(String ^ skinName, String ^ fileName)
      {
         m_Command = "StoreServerSkinFile";
         CheckParam(skinName);
         CheckParam(fileName);
         m_Command += " \"" + skinName + "\" FILE \"" + fileName + "\" DELETE";
         SendCmd(m_Command);
      }

      DataSet ^ ListClusterSkins()
      {
         m_Command = "ListClusterSkins";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void CreateClusterSkin(String ^ skinName)
      {
         m_Command = "CreateClusterSkin";
         CheckParam(skinName);
         m_Command += " \"" + skinName + "\"";
         SendCmd(m_Command);
      }

      void RenameClusterSkin(String ^ skinName, String ^ newName)
      {
         m_Command = "RenameClusterSkin";
         CheckParam(skinName);
         CheckParam(newName);
         m_Command += " \"" + skinName + "\" INTO \"" + newName + "\"";
         SendCmd(m_Command);
      }

      void DeleteClusterSkin(String ^ skinName)
      {
         m_Command = "DeleteClusterSkin";
         CheckParam(skinName);
         m_Command += " \"" + skinName + "\"";
         SendCmd(m_Command);
      }

      DataSet ^ ListClusterSkinFiles(String ^ skinName)
      {
         m_Command = "ListClusterSkinFiles";
         CheckParam(skinName);
         m_Command += " \"" + skinName + "\"";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      DataSet ^ ReadClusterSkinFile(String ^ skinName, String ^ fileName)
      {
         m_Command = "ReadClusterSkinFile";
         CheckParam(skinName);
         CheckParam(fileName);
         m_Command += " \"" + skinName + "\" FILE \"" + fileName + "\"";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void StoreClusterSkinFile(String ^ skinName, String ^ fileName, String ^ in_data)
      {
         m_Command = "StoreClusterSkinFile";
         CheckParam(skinName);
         CheckParam(fileName);
         CheckParam(in_data);
         m_Command += " \"" + skinName + "\" FILE \"" + fileName + "\" DATA \"" + in_data + "\"";
         SendCmd(m_Command);
      }

      void DeleteClusterSkinFile(String ^ skinName, String ^ fileName)
      {
         m_Command = "StoreClusterSkinFile";
         CheckParam(skinName);
         CheckParam(fileName);
         m_Command += " \"" + skinName + "\" FILE \"" + fileName + "\" DELETE";
         SendCmd(m_Command);
      }

      DataSet ^ ListWebUserInterface(String ^ domainName, String ^ path)
      {
         m_Command = "ListWebUserInterface";
         CheckParam(domainName);
         m_Command += " " + domainName;
         if(path->Length)
            m_Command += " FILE \"" + path + "\"";
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      String ^ GetWebUserInterface(String ^ domainName, String ^ fileName)
      {
         m_Command = "GetWebUserInterface";
         CheckParam(domainName);
         CheckParam(fileName);
         m_Command += " " + domainName + " FILE \"" + fileName + "\"";
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords())->StringValue;
      }

      void PutWebUserInterface(String ^ domainName, String ^ fileName, String ^ in_data)
      {
         m_Command = "PutWebUserInterface";
         CheckParam(domainName);
         CheckParam(fileName);
         CheckParam(in_data);
         m_Command += " " + domainName + " FILE \"" + fileName + "\" DATA " + in_data;
         StreamSend(m_Command);
         ParseResponse();
      }

      void DeleteWebUserInterface(String ^ domainName, String ^ fileName)
      {
         m_Command = "DeleteWebUserInterface";
         CheckParam(domainName);
         CheckParam(fileName);
         m_Command += " " + domainName + " FILE \"" + fileName + "\"";
         StreamSend(m_Command);
         ParseResponse();
      }

      void ClearWebUserCache(String ^ domainName)
      {
         m_Command = "ClearWebUserCache";
         CheckParam(domainName);
         m_Command += " " + domainName;
         StreamSend(m_Command);
         ParseResponse();
      }

      String ^ CreateWebUserSession(String ^ accountName, String ^ ipAddress, String ^ skin)
      {
         m_Command = "CreateWebUserSession";
         CheckParam(accountName);
         CheckParam(ipAddress);
         m_Command += " " + accountName + " ADDRESS " + ipAddress;
         if(skin->Length)
            m_Command += " SKIN \"" + skin + "\"";
         SendCmd(m_Command);
         return ParseWords(GetWords())->StringValue;
      }

      DataSet ^ GetWebUserSession(String ^ sessionID)
      {
         m_Command = "GetWebUserSession";
         CheckParam(sessionID);
         m_Command += " " + sessionID;
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void KillWebUserSession(String ^ sessionID)
      {
         m_Command = "KillWebUserSession";
         CheckParam(sessionID);
         m_Command += " " + sessionID;
         SendCmd(m_Command);
      }

      DataSet ^ GetModule(String ^ moduleName)
      {
         m_Command = "GetModule";
         CheckParam(moduleName);
         m_Command += " " + moduleName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void UpdateModule(String ^ moduleName, DataSet ^ settings)
      {
         m_Command = "UpdateModule";
         CheckParam(moduleName);
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + moduleName + " " + PrintWords(settings);
         StreamSend(m_Command);
         ParseResponse();
      }

      void SetModule(String ^ moduleName, DataSet ^ settings)
      {
         m_Command = "SetModule";
         CheckParam(moduleName);
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + moduleName + " " + PrintWords(settings);
         StreamSend(m_Command);
         ParseResponse();
      }

      String ^ GetLANIPs()
      {
         m_Command = "GetLANIPs";
         SendCmd(m_Command);
         return ParseWords(GetWords())->StringValue;
      }

      String ^ GetBlacklistedIPs()
      {
         m_Command = "GetBlacklistedIPs";
         SendCmd(m_Command);
         return ParseWords(GetWords())->StringValue;
      }

      String ^ GetClientIPs()
      {
         m_Command = "GetClientIPs";
         SendCmd(m_Command);
         return ParseWords(GetWords())->StringValue;
      }

      String ^ GetWhiteHoleIPs()
      {
         m_Command = "GetWhiteHoleIPs";
         SendCmd(m_Command);
         return ParseWords(GetWords())->StringValue;
      }

      DataSet ^ GetProtection()
      {
         m_Command = "GetProtection";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      DataSet ^ GetBanned()
      {
         m_Command = "GetBanned";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetBlacklistedIPs(String ^ addresses)
      {
         m_Command = "SetBlacklistedIPs";
         CheckParam(addresses);
         m_Command += " \"" + addresses + "\"";
         SendCmd(m_Command);
      }

      void SetLANIPs(String ^ addresses)
      {
         m_Command = "SetLANIPs";
         CheckParam(addresses);
         m_Command += " \"" + addresses + "\"";
         SendCmd(m_Command);
      }

      void SetClientIPs(String ^ addresses)
      {
         m_Command = "SetClientIPs";
         CheckParam(addresses);
         m_Command += " \"" + addresses + "\"";
         SendCmd(m_Command);
      }

      void SetWhiteHoleIPs(String ^ addresses)
      {
         m_Command = "SetWhiteHoleIPs";
         CheckParam(addresses);
         m_Command += " \"" + addresses + "\"";
         SendCmd(m_Command);
      }

      void SetProtection(DataSet ^ settings)
      {
         m_Command = "SetProtection";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      void SetBanned(DataSet ^ settings)
      {
         m_Command = "SetBanned";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      String ^ GetClusterBlacklistedIPs()
      {
         m_Command = "GetClusterBlacklistedIPs";
         SendCmd(m_Command);
         return ParseWords(GetWords())->StringValue;
      }

      String ^ GetClusterLANIPs()
      {
         m_Command = "GetClusterLANIPs";
         SendCmd(m_Command);
         return ParseWords(GetWords())->StringValue;
      }

      String ^ GetClusterClientIPs()
      {
         m_Command = "GetClusterClientIPs";
         SendCmd(m_Command);
         return ParseWords(GetWords())->StringValue;
      }

      String ^ GetClusterWhiteHoleIPs()
      {
         m_Command = "GetClusterWhiteHoleIPs";
         SendCmd(m_Command);
         return ParseWords(GetWords())->StringValue;
      }

      DataSet ^ GetClusterProtection()
      {
         m_Command = "GetClusterProtection";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      DataSet ^ GetClusterBanned()
      {
         m_Command = "GetClusterBanned";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetClusterBlacklistedIPs(String ^ addresses)
      {
         m_Command = "SetClusterBlacklistedIPs";
         CheckParam(addresses);
         m_Command += " \"" + addresses + "\"";
         SendCmd(m_Command);
      }

      void SetClusterClientIPs(String ^ addresses)
      {
         m_Command = "SetClusterClientIPs";
         CheckParam(addresses);
         m_Command += " \"" + addresses + "\"";
         SendCmd(m_Command);
      }

      void SetClusterLANIPs(String ^ addresses)
      {
         m_Command = "SetClusterLANIPs";
         CheckParam(addresses);
         m_Command += " \"" + addresses + "\"";
         SendCmd(m_Command);
      }

      void SetClusterWhiteHoleIPs(String ^ addresses)
      {
         m_Command = "SetClusterWhiteHoleIPs";
         CheckParam(addresses);
         m_Command += " \"" + addresses + "\"";
         SendCmd(m_Command);
      }

      void SetClusterProtection(DataSet ^ settings)
      {
         m_Command = "SetClusterProtection";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      DataSet ^ GetServerRules()
      {
         m_Command = "GetServerRules";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetServerRules(DataSet ^ rules)
      {
         m_Command = "SetServerRules";
         CheckParam(rules, DataSet::DataSetType::Array);
         m_Command += " " + PrintWords(rules);
         SendCmd(m_Command);
      }

      DataSet ^ GetClusterRules()
      {
         m_Command = "GetClusterRules";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetClusterRules(DataSet ^ rules)
      {
         m_Command = "SetClusterRules";
         CheckParam(rules, DataSet::DataSetType::Array);
         m_Command += " " + PrintWords(rules);
         SendCmd(m_Command);
      }

      void RefreshOSData()
      {
         m_Command = "RefreshOSData";
         StreamSend(m_Command);
         ParseResponse();
      }

      String ^ GetRouterTable()
      {
         m_Command = "GetRouterTable";
         SendCmd(m_Command);
         return ParseWords(GetWords())->StringValue;
      }

      void SetRouterTable(String ^ table)
      {
         m_Command = "SetRouterTable";
         CheckParam(table);
         m_Command += " " + PrintWords(table);
         SendCmd(m_Command);
      }

      DataSet ^ GetRouterSettings()
      {
         m_Command = "GetRouterSettings";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetRouterSettings(DataSet ^ settings)
      {
         m_Command = "SetRouterSettings";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      String ^ GetClusterRouterTable()
      {
         m_Command = "GetClusterRouterTable";
         SendCmd(m_Command);
         return ParseWords(GetWords())->StringValue;
      }

      void SetClusterRouterTable(String ^ table)
      {
         m_Command = "SetClusterRouterTable";
         CheckParam(table);
         m_Command += " " + PrintWords(table);
         SendCmd(m_Command);
      }

      DataSet ^ GetServerIntercept()
      {
         m_Command = "GetServerIntercept";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetServerIntercept(DataSet ^ settings)
      {
         m_Command = "SetServerIntercept";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      DataSet ^ GetClusterIntercept()
      {
         m_Command = "GetClusterIntercept";
         SendCmd(m_Command);
         return ParseWords(GetWords());
      }

      void SetClusterIntercept(DataSet ^ settings)
      {
         m_Command = "SetClusterIntercept";
         CheckParam(settings, DataSet::DataSetType::Dictionary);
         m_Command += " " + PrintWords(settings);
         SendCmd(m_Command);
      }

      DataSet ^ Route(String ^ address, bool mail)
      {
         m_Command = "Route";
         CheckParam(address);
         m_Command += " " + address;
         if(mail)
            m_Command += " mail";
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      String ^ GetSNMPElement(String ^ elemName)
      {
         m_Command = "GetSNMPElement";
         CheckParam(elemName);
         m_Command += " " + elemName;
         SendCmd(m_Command);
         return ParseWords(GetWords())->StringValue;
      }

      void Shutdown()
      {
         m_Command = "Shutdown";
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ GetAccountStat(String ^ accountName)
      {
         m_Command = "GetAccountStat";
         CheckParam(accountName);
         m_Command += " " + accountName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void ResetAccountStat(String ^ accountName)
      {
         m_Command = "ResetAccountStat";
         CheckParam(accountName);
         m_Command += " " + accountName;
         StreamSend(m_Command);
         ParseResponse();
      }

      DataSet ^ GetDomainStat(String ^ domainName)
      {
         m_Command = "GetDomainStat";
         CheckParam(domainName);
         m_Command += " " + domainName;
         StreamSend(m_Command);
         ParseResponse();
         return ParseWords(GetWords());
      }

      void ResetDomainStat(String ^ domainName)
      {
         m_Command = "ResetDomainStat";
         CheckParam(domainName);
         m_Command += " " + domainName;
         StreamSend(m_Command);
         ParseResponse();
      }

      void WriteLog(int level, String ^ message)
      {
         m_Command = "WriteLog";
         CheckParam(message);
         m_Command += " " + level.ToString() + " " + PrintWords(EncodeString(message));
         StreamSend(m_Command);
         ParseResponse();
      }

      void ReleaseSMTPQueue(String ^ queueName)
      {
         m_Command = "ReleaseSMTPQueue";
         CheckParam(queueName);
         m_Command += " " + queueName;
         StreamSend(m_Command);
         ParseResponse();
      }

      void RejectQueueMessage(String ^ messageID, String ^ errorText)
      {
         m_Command = "RejectQueueMessage";
         CheckParam(messageID);
         m_Command += " " + messageID;
         if(errorText->Length)
            m_Command += " " + PrintWords(errorText);
         StreamSend(m_Command);
         ParseResponse();
      }

      String ^ GetCurrentController()
      {
         m_Command = "GetCurrentController";
         SendCmd(m_Command);
         return ParseWords(GetWords())->StringValue;
      }

      String ^ GetTempClientIPs()
      {
         m_Command = "GetTempClientIPs";
         SendCmd(m_Command);
         return ParseWords(GetWords())->StringValue;
      }

      void RemoveAccountSubset(String ^ accountName, String ^ subset)
      {
         m_Command = "RemoveAccountSubset";
         CheckParam(accountName);
         CheckParam(subset);
         m_Command += " " + accountName + " SUBSET " + PrintWords(subset);
         StreamSend(m_Command);
         ParseResponse();
      }
   };
}
